/*
  konq_rellinks.cpp
  A plugin to display document relation links in konquerors menus/toolbars

  begin:  Sun Jan 20 15:34:41 CET 2002
  author: Anders Lund, anders@alweb.dk, copyright 2002
  new-begin: Apr 1st CET 2003
  author: Franck Qulain, shift@free.fr, copyright 2003

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
*/

#include <dom/dom_doc.h>
#include <dom/dom_element.h>
#include <dom/dom_string.h>
#include <dom/html_document.h>
#include <kaction.h>
#include <kgenericfactory.h>
#include <khtml_part.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kshortcut.h>
#include <ktoolbar.h>
#include <kurl.h>
#include <qmap.h>
#include <ksimpleconfig.h>

#include "konq_rellinks.h"

const QString KonqRellinks::CONF_ALWAYS = "always";
const QString KonqRellinks::CONF_NEVER = "never";
const QString KonqRellinks::CONF_WHEN_NECESSARY = "when necessary";

typedef KGenericFactory<KonqRellinks> KonqRellinksFactory;
K_EXPORT_COMPONENT_FACTORY( libkonqrellinks, KonqRellinksFactory("rellinks") );

/* Constructor */
KonqRellinks::KonqRellinks(QObject *parent, const char *name, const QStringList &)
  : KParts::Plugin( parent, name )
{
  // Config
  kconfig = new KSimpleConfig("konq_rellinksrc");
  listitem_conf.append(CONF_ALWAYS);
  listitem_conf.append(CONF_WHEN_NECESSARY);
  listitem_conf.append(CONF_NEVER);
  kconfig->setGroup("General");
  QString mode = kconfig->readEntry("mode", QString::null);
  if (mode == QString::null) {
	mode = CONF_WHEN_NECESSARY;
	kconfig->writeEntry("mode", mode);
	kconfig->sync();
	kconfig->reparseConfiguration();
  }
  int mode_index = listitem_conf.findIndex(mode);

  m_conf = new KSelectAction( i18n("Configure"), "configure", KShortcut(), actionCollection(), "rellinks_config" );
  QStringList listitem;
  listitem.append(i18n("always"));
  listitem.append(i18n("when necessary"));
  listitem.append(i18n("never"));
  m_conf->setItems(listitem);

  connect(m_conf->popupMenu(), SIGNAL( activated( int ) ), this, SLOT( changeConf( int ) ) );

  m_conf->setCurrentItem(mode_index);


  m_url = QString::null;

  // Navigation links
  kaction_map["home"] =  new KAction( i18n("Top"), "2uparrow", KShortcut(), this, SLOT(goHome()), actionCollection(), "rellinks_top" );
  kaction_map["home"]->setWhatsThis( i18n("<p>This link references a home page or the top of some hierarchy.</p>") );

  kaction_map["up"] =  new KAction( i18n("Up"), "1uparrow", KShortcut(), this, SLOT(goUp()), actionCollection(), "rellinks_up" );
  kaction_map["up"]->setWhatsThis( i18n("<p>This link references the immediate parent of the current document.</p>") );

  kaction_map["start"] =  new KAction( i18n("First"), "2leftarrow", KShortcut(), this,  SLOT(goFirst()), actionCollection(), "rellinks_first" );
  kaction_map["start"]->setWhatsThis( i18n("<p>This link type tells search engines which document is considered by the author to be the starting point of the collection.</p>") );

  kaction_map["prev"] =  new KAction( i18n("Previous"), "1leftarrow", KShortcut(), this,  SLOT(goPrevious()), actionCollection(), "rellinks_previous" );
  kaction_map["prev"]->setWhatsThis( i18n("<p>This link references the previous document in an ordered series of documents.</p>") );

  kaction_map["next"] =  new KAction( i18n("Next"), "1rightarrow", KShortcut(), this,  SLOT(goNext()), actionCollection(), "rellinks_next" );
  kaction_map["next"]->setWhatsThis( i18n("<p>This link references the next document in an ordered series of documents.</p>") );

  kaction_map["last"] =  new KAction( i18n("Last"), "2rightarrow", KShortcut(), this,  SLOT(goLast()), actionCollection(), "rellinks_last" );
  kaction_map["last"]->setWhatsThis( i18n("<p>This link references the end of a sequence of documents.</p>") );


  // Document structure links
  m_document = new KActionMenu( i18n("Document"),  "contents2", actionCollection(), "rellinks_document" );
  m_document->setWhatsThis( i18n("<p>This menu contains the links refering the document information.</p>") );

  kaction_map["contents"] = new KAction( i18n("Table of Contents"), "contents", KShortcut(),  this,  SLOT(goContents()), actionCollection(), "rellinks_toc" );
  m_document->insert(kaction_map["contents"]);
  kaction_map["contents"]->setWhatsThis( i18n("<p>This link references the table of contents.</p>") );

  kactionmenu_map["chapter"] = new KActionMenu( i18n("Chapters"), "fileopen", actionCollection(), "rellinks_chapters" );
  m_document->insert(kactionmenu_map["chapter"]);
  connect( kactionmenu_map["chapter"]->popupMenu(), SIGNAL( activated( int ) ), this, SLOT(goChapter(int)));
  kactionmenu_map["chapter"]->setWhatsThis( i18n("<p>This menu references the chapters of the document.</p>") );


  kactionmenu_map["section"] = new KActionMenu( i18n("Sections"), "fileopen", actionCollection(), "rellinks_sections" );
  m_document->insert(kactionmenu_map["section"]);
  connect( kactionmenu_map["section"]->popupMenu(), SIGNAL( activated( int ) ), this, SLOT( goSection( int ) ) );
  kactionmenu_map["section"]->setWhatsThis( i18n("<p>This menu references the sections of the document.</p>") );


  kactionmenu_map["subsection"] = new KActionMenu( i18n("Subsections"), "fileopen", actionCollection(), "rellinks_subsections" );
  m_document->insert(kactionmenu_map["subsection"]);
  connect( kactionmenu_map["subsection"]->popupMenu(), SIGNAL( activated( int ) ), this, SLOT( goSubsection( int ) ) );
  kactionmenu_map["subsection"]->setWhatsThis( i18n("<p>This menu references the subsections of the document.</p>") );

  kactionmenu_map["appendice"] = new KActionMenu( i18n("Appendices"), "edit", actionCollection(), "rellinks_appendices" );
  m_document->insert(kactionmenu_map["appendice"]);
  connect( kactionmenu_map["appendice"]->popupMenu(), SIGNAL( activated( int ) ), this, SLOT( goAppendice( int ) ) );
  kactionmenu_map["appendice"]->setWhatsThis( i18n("<p>This link references the appendice.</p>") );

  kaction_map["glossary"] = new KAction( i18n("Glossary"), "flag", KShortcut(), this, SLOT(goGlossary()), actionCollection(), "rellinks_glossary" );
  m_document->insert(kaction_map["glossary"]);
  kaction_map["glossary"]->setWhatsThis( i18n("<p>This link references the glossary.</p>") );

  kaction_map["index"] = new KAction( i18n("Index"), "info", KShortcut(), this, SLOT(goIndex()), actionCollection(), "rellinks_index" );
  m_document->insert(kaction_map["index"]);
  kaction_map["index"]->setWhatsThis( i18n("<p>This link references the index.</p>") );

  // Other links
  m_more  = new KActionMenu( i18n("More"), "misc", actionCollection(), "rellinks_more" );
  m_more->setWhatsThis( i18n("<p>This menu contains other important links.</p>") );

  kaction_map["help"] = new KAction( i18n("Help"), "help", KShortcut(), this, SLOT(goHelp()), actionCollection(), "rellinks_help" );
  m_more->insert(kaction_map["help"]);
  kaction_map["help"]->setWhatsThis( i18n("<p>This link references the help.</p>") );

  kaction_map["search"]  = new KAction( i18n("Search"), "filefind", KShortcut(), this, SLOT(goSearch()), actionCollection(), "rellinks_search" );
  m_more->insert(kaction_map["search"]);
  kaction_map["search"]->setWhatsThis( i18n("<p>This link references the search.</p>") );

  kaction_map["made"]  = new KAction( i18n("Authors"), "mail_new", KShortcut(), this, SLOT(goAuthor()), actionCollection(), "rellinks_authors" );
  m_more->insert(kaction_map["made"]);
  kaction_map["made"]->setWhatsThis( i18n("<p>This link references the author.</p>") );

  kaction_map["copyright"]   = new KAction( i18n("Copyright"), "signature", KShortcut(), this, SLOT(goCopyright()), actionCollection(), "rellinks_copyright" );
  m_more->insert(kaction_map["copyright"]);
  kaction_map["copyright"]->setWhatsThis( i18n("<p>This link references the copyright.</p>") );

  kactionmenu_map["bookmark"] = new KActionMenu( i18n("Bookmarks"), "bookmark_folder", actionCollection(), "rellinks_bookmarks" );
  m_more->insert(kactionmenu_map["bookmark"]);
  kactionmenu_map["bookmark"]->setWhatsThis( i18n("<p>This menu references the bookmarks.</p>") );
  connect( kactionmenu_map["bookmark"]->popupMenu(), SIGNAL( activated( int ) ), this, SLOT( goBookmark( int ) ) );

  kactionmenu_map["alternate"] = new KActionMenu( i18n("Other Versions"), "attach", actionCollection(), "rellinks_other_versions" );
  m_more->insert(kactionmenu_map["alternate"]);
  kactionmenu_map["alternate"]->setWhatsThis( i18n("<p>This link references the alternate versions of this document.</p>") );
  connect( kactionmenu_map["alternate"]->popupMenu(), SIGNAL( activated( int ) ), this, SLOT( goAlternate( int ) ) );

  // Unclassified menu
  kactionmenu_map["unclassified"] = new KActionMenu( i18n("Unclassified"), "rellinks", actionCollection(), "rellinks_links" );
  kactionmenu_map["unclassified"]->setWhatsThis( i18n("<p>Unclassified links.</p>") );
  connect( kactionmenu_map["unclassified"]->popupMenu(), SIGNAL( activated( int ) ), this, SLOT( goAllElements( int ) ) );


  // We desactivate all the possible actions
  desactivateAll();

  // When the rendering of the HTML is done, we update the site navigation bar
  KHTMLPart *part = dynamic_cast<KHTMLPart *>(parent);
  if (!part) return;
  connect( part, SIGNAL( completed() ), this, SLOT( updateToolbar() ) );
  //connect( part, SIGNAL(  createPart ( QWidget *, const char *, QObject *, const char *, const QString , QString, QStringList , const QStringList ) ), this, SLOT( desactivateAll() ) );
}

/* Destructor */
KonqRellinks::~KonqRellinks()
{
}

/* Update the site navigation bar */
void KonqRellinks::updateToolbar()
{

  // If we have a part
  KHTMLPart *part = dynamic_cast<KHTMLPart *>(parent());
  if (!part) return;
  // and needs update
  if ( m_url == part->htmlDocument().URL().string() ) {
    return;
  }

  // Url of the document to test if we need update next time
  m_url = part->htmlDocument().URL().string();

  // We desactivate all
  desactivateAll();

  // get a list of LINK nodes in document
  DOM::NodeList linkNodes = part->document().getElementsByTagName( "LINK" );


  bool showBar = false;

  for ( unsigned int i=0; i < linkNodes.length(); i++ ) {
    // create a entry for each one
    DOM::Element e( linkNodes.item( i ) );


    QString rel = e.getAttribute( "REL" ).string();
    rel = rel.simplifyWhiteSpace();
    if (rel.isEmpty()) {
      // If the "rel" attribut is null then use the "rev" attribute
      // TODO : Make something to distinct this ?
      rel =  e.getAttribute( "REV" ).string();
      rel = rel.simplifyWhiteSpace();
    }


    if ( rel.lower() == "stylesheet" ) continue; // skip CSS style sheets
    QString title = e.getAttribute( "TITLE" ).string();
    QString hreflang = e.getAttribute( "HREFLANG" ).string();

    KURL ref( part->htmlDocument().completeURL( e.getAttribute( "HREF" ) ).string() );
    if ( title.isEmpty() ) title = ref.prettyURL();

    QString lrel = rel.lower();

    // Other names used for the same elements
    if (lrel == "top" || lrel == "origin") lrel = "home";
    if (lrel == "begin" || lrel == "first") lrel = "start";
    if (lrel == "previous") lrel = "prev";
    if (lrel == "end") lrel = "last";
    if (lrel == "toc") lrel = "contents";
    if (lrel == "find") lrel = "search";
    if (lrel == "alternative stylesheet") lrel = "alternate stylesheet";
    // TODO : authors need to be a menu ?
    if (lrel == "authors" || lrel == "author" ) lrel = "made";

    // Activation of "Document" menu ?
    if (lrel == "contents" || lrel == "glossary" || lrel == "index" || lrel == "appendices") {
       m_document->setEnabled(true);
    }

    // Activation of "More" menu ?
    if (lrel == "help" || lrel == "search" || lrel == "made" || lrel == "copyright" ) {
       m_document->setEnabled(true);
    }

    if (lrel == "bookmark") {
      int id = kactionmenu_map[lrel]->popupMenu()->insertItem( title );
      m_more->setEnabled(true);
      kactionmenu_map[lrel]->setEnabled(true);
      menu_url_map[lrel][id] = ref;

    } else if (lrel == "alternate") {
      int id = kactionmenu_map[lrel]->popupMenu()->insertItem( title );
      m_more->setEnabled(true);
      kactionmenu_map[lrel]->setEnabled(true);
      menu_url_map[lrel][id] = ref;

    } else if (lrel == "appendice") {
      int id = kactionmenu_map[lrel]->popupMenu()->insertItem( title );
      kactionmenu_map[lrel]->setEnabled(true);
      m_document->setEnabled(true);
      menu_url_map[lrel][id] = ref;

    } else if (lrel == "chapter") {
      int id = kactionmenu_map[lrel]->popupMenu()->insertItem( title );
      m_document->setEnabled(true);
      kactionmenu_map[lrel]->setEnabled(true);
      menu_url_map[lrel][id] = ref;

    } else if (lrel == "section") {
      int id = kactionmenu_map[lrel]->popupMenu()->insertItem( title );
      m_document->setEnabled(true);
      kactionmenu_map[lrel]->setEnabled(true);
      menu_url_map[lrel][id] = ref;

    } else if (lrel == "subsection") {
      int id = kactionmenu_map[lrel]->popupMenu()->insertItem( title );
      m_document->setEnabled(true);
      kactionmenu_map[lrel]->setEnabled(true);
      menu_url_map[lrel][id] = ref;

    } else if (lrel == "stylesheet") {
      continue; // skip CSS style sheets

    } else if (lrel == "alternate stylesheet") {
      continue; // skip CSS style sheets

    } else if (lrel == "script") {
      continue; // skip script

    } else if (lrel == "icon" || lrel == "shortcut icon") {
      continue; // skip favicon

    } else if (lrel == "prefetch") {
      continue; // skip prefetch

    } else {
      // It is a unique action
      url_map[lrel] = ref;
      if (kaction_map[lrel]) {
        kaction_map[lrel]->setEnabled(true);
        // Tooltip
        if (hreflang.isEmpty()) {
          kaction_map[lrel]->setToolTip( title );
        } else {
          kaction_map[lrel]->setToolTip( title + " [" + hreflang + "]");
        }
      } else {
        // For the moment all the elements are reference in a separated menu
        // TODO : reference the unknown ?
        int id = kactionmenu_map["unclassified"]->popupMenu()->insertItem( lrel + " : " + title );
        kactionmenu_map["unclassified"]->setEnabled(true);
        menu_url_map["unclassified"][id] = ref;
      }


    }

    showBar = true;
  }


  // Get the conf
  kconfig->setGroup("General");
  QString mode = kconfig->readEntry("mode", "when necessary");
  KToolBar *bar = dynamic_cast<KToolBar *>(part->widget()->topLevelWidget()->child("Document Relations Toolbar", "KToolBar"));

  if (mode == CONF_ALWAYS) {
	bar->show();
  } else if (mode == CONF_NEVER) {
	bar->hide();
  /*
  } else if (mode == CONF_XHEN_NECESSARY) {

	if (showBar) {
	// We must show the toolbar
	bar->show();
	} else {
	// We must hide it
	bar->hide();
	}
     */
  } else {
	if (showBar) {
	// We must show the toolbar
	bar->show();
	} else {
	// We must hide it
	bar->hide();
	}
  }


}

/** Simple links */

void KonqRellinks::goToLink(QString link)
{
  // have the KHTML part open it
  KHTMLPart *part = dynamic_cast<KHTMLPart *>(parent());
  if (!part) return;

  KURL url = url_map[link];

  // Add base url if not valid
  if (url.isValid()) {
    part->browserExtension()->openURLRequest(url);
  } else {
    KURL baseURL = part->baseURL();
    QString endURL = url.prettyURL();
    KURL realURL = KURL(baseURL, endURL);
    part->browserExtension()->openURLRequest(realURL);
  }

}

void KonqRellinks::goHome()
{
  goToLink("home");
}

void KonqRellinks::goUp()
{
  goToLink("up");
}

void KonqRellinks::goFirst()
{
  goToLink("first");
}

void KonqRellinks::goPrevious()
{
  goToLink("prev");
}

void KonqRellinks::goNext()
{
  goToLink("next");
}

void KonqRellinks::goLast()
{
  goToLink("last");
}

void KonqRellinks::goContents()
{
  goToLink("contents");
}

void KonqRellinks::goIndex()
{
  goToLink("index");
}

void KonqRellinks::goGlossary()
{
  goToLink("glossary");
}

void KonqRellinks::goHelp()
{
  goToLink("help");
}

void KonqRellinks::goSearch()
{
  goToLink("search");
}

void KonqRellinks::goAuthor()
{
  goToLink("made");
}


void KonqRellinks::goCopyright()
{
  goToLink("copyright");
}


/** Menu links */

void KonqRellinks::goToLink(QString link, int id)
{
  // have the KHTML part open it
  KHTMLPart *part = dynamic_cast<KHTMLPart *>(parent());
  if (!part) return;

  KURL url = menu_url_map[link][id];

  // Add base url if not valid
  if (url.isValid()) {
    part->browserExtension()->openURLRequest(url);
  } else {
    KURL baseURL = part->baseURL();
    QString endURL = url.prettyURL();
    KURL realURL = KURL(baseURL, endURL);
    part->browserExtension()->openURLRequest(realURL);
  }

}

void KonqRellinks::goBookmark(int id)
{
  goToLink("bookmark", id);
}

void KonqRellinks::goChapter(int id)
{
  goToLink("chapter", id);
}

void KonqRellinks::goSection(int id)
{
  goToLink("section", id);
}

void KonqRellinks::goSubsection(int id)
{
  goToLink("subsection", id);
}

void KonqRellinks::goAppendice(int id)
{
  goToLink("appendice", id);
}

void KonqRellinks::goAlternate(int id)
{
  goToLink("alternate", id);
}

void KonqRellinks::goAllElements(int id)
{
  goToLink("unclassified", id);
}

void KonqRellinks::desactivateAll()
{
  typedef QMap<QString, KAction*> KactionMap;
  typedef QMap<QString, KActionMenu*> KactionMenuMap;

  url_map.clear();
  menu_url_map.clear();

  // Clear actions
  KactionMap::Iterator it;
  for ( it = kaction_map.begin(); it != kaction_map.end(); ++it ) {
      // If I don't test it crash :(
      if (kaction_map[it.key()]) {
        kaction_map[it.key()]->setEnabled(false);
        kaction_map[it.key()]->setToolTip(kaction_map[it.key()]->text());
      }
  }

  // Clear actions
  KactionMenuMap::Iterator itmenu;
  for ( itmenu = kactionmenu_map.begin(); itmenu != kactionmenu_map.end(); ++itmenu ) {
      // If I don't test it crash :(
      if (kactionmenu_map[itmenu.key()]) {
        kactionmenu_map[itmenu.key()]->popupMenu()->clear();
        kactionmenu_map[itmenu.key()]->setEnabled(false);
        kactionmenu_map[itmenu.key()]->setToolTip(kactionmenu_map[itmenu.key()]->text());
      }
  }

  // Unactivate menus
  m_more->setEnabled(false);
  m_document->setEnabled(false);

}

void KonqRellinks::changeConf(int id)
{
  // Get the conf
  kconfig->setGroup("General");
  kconfig->writeEntry("mode", listitem_conf[id]);
  kconfig->sync();
  kconfig->reparseConfiguration();

  m_url = QString::null;
  this->updateToolbar();
}


#include "konq_rellinks.moc"
